<?php

namespace HashElements\Modules\TickerModule\Widgets;

// Elementor Classes
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use HashElements\Group_Control_Query;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class TickerModule extends Widget_Base {

    /** Widget Name */
    public function get_name() {
        return 'he-ticker-module';
    }

    /** Widget Title */
    public function get_title() {
        return esc_html__('Ticker Module', 'hash-elements');
    }

    /** Icon */
    public function get_icon() {
        return 'he-ticker-module he-news-modules';
    }

    /** Category */
    public function get_categories() {
        return ['he-magazine-elements'];
    }

    /** Controls */
    protected function register_controls() {

        $this->start_controls_section(
            'ticker', [
                'label' => esc_html__('Ticker Block', 'hash-elements'),
            ]
        );

        $this->add_control(
            'ticker_title', [
                'label' => esc_html__('Title', 'hash-elements'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Latest Posts', 'hash-elements'),
                'placeholder' => esc_html__('Type your title here', 'hash-elements'),
                'label_block' => true
            ]
        );

        $this->add_control(
            'autoplay', [
                'label' => esc_html__('Autoplay', 'hash-elements'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'hash-elements'),
                'label_off' => esc_html__('No', 'hash-elements'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'ticker_pause_duration', [
                'label' => esc_html__('Ticker Pause Duration', 'hash-elements'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 10,
                        'step' => 1
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'condition' => [
                    'autoplay' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_post_query', [
                'label' => esc_html__('Content Filter', 'hash-elements'),
            ]
        );

        $this->add_group_control(
            Group_Control_Query::get_type(), [
                'name' => 'posts',
                'label' => esc_html__('Posts', 'hash-elements'),
            ]
        );

        $this->add_control(
            'ticker_post_count', [
                'label' => esc_html__('No of Posts', 'hash-elements'),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 10,
                'step' => 1,
                'default' => 5,
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'ticker_title_style', [
                'label' => esc_html__('Title', 'hash-elements'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(), [
                'name' => 'ticker_title_typography',
                'label' => esc_html__('Typography', 'hash-elements'),
                'selector' => '{{WRAPPER}} .he-ticker .he-ticker-title'
            ]
        );

        $this->add_control(
            'ticker_title_bg_color', [
                'label' => esc_html__('Background Color', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker .he-ticker-title' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .he-ticker .he-ticker-title:after' => 'border-color: transparent transparent transparent {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'ticker_title_color', [
                'label' => esc_html__('Text Color', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker .he-ticker-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'ticker_content_style', [
                'label' => esc_html__('Content', 'hash-elements'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(), [
                'name' => 'ticker_content_typography',
                'label' => esc_html__('Typography', 'hash-elements'),
                'selector' => '{{WRAPPER}} .he-ticker .owl-item a'
            ]
        );

        $this->add_control(
            'ticker_content_bg_color', [
                'label' => esc_html__('Background Color', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'ticker_content_color', [
                'label' => esc_html__('Text Color', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker .owl-item a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'navigation_style', [
                'label' => esc_html__('Navigation', 'hash-elements'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs(
            'nav_style_tabs'
        );

        $this->start_controls_tab(
            'nav_normal_tab', [
                'label' => esc_html__('Normal', 'hash-elements'),
            ]
        );

        $this->add_control(
            'nav_normal_bg_color', [
                'label' => esc_html__('Background Color', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker .owl-carousel .owl-nav button' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'nav_icon_normal_color', [
                'label' => esc_html__('Icon Color', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker .owl-carousel .owl-nav button' => 'color: {{VALUE}}'
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'nav_hover_tab', [
                'label' => esc_html__('Hover', 'hash-elements'),
            ]
        );

        $this->add_control(
            'nav_hover_bg_color', [
                'label' => esc_html__('Background Color (Hover)', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker .owl-carousel .owl-nav button:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'nav_icon_hover_color', [
                'label' => esc_html__('Icon Color (Hover)', 'hash-elements'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .he-ticker .owl-carousel .owl-nav button:hover' => 'color: {{VALUE}}'
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    /** Render Layout */
    protected function render() {
        $settings = $this->get_settings_for_display();

        $ticker_pause = $settings['ticker_pause_duration']['size'];

        $parameters = array(
            'pause' => intval($ticker_pause),
            'autoplay' => $settings['autoplay'] == 'yes' ? true : false,
        );

        $parameters_json = json_encode($parameters);


        $args = $this->query_args();
        $query = new \WP_Query($args);
        if ($query->have_posts()):
            ?>
            <div class="he-ticker">
                <div class="he-container">
                    <span class="he-ticker-title">
                        <?php
                        $ticker_title = isset($settings['ticker_title']) ? $settings['ticker_title'] : null;
                        if ($ticker_title) {
                            echo esc_html($ticker_title);
                        }
                        ?>
                    </span>
                    <div class="owl-carousel" data-params='<?php echo esc_attr($parameters_json); ?>'>
                        <?php
                        while ($query->have_posts()):
                            $query->the_post();
                            echo '<a href="' . esc_url(get_permalink()) . '">' . esc_html(get_the_title()) . '</a>';
                        endwhile;
                        wp_reset_postdata();
                        ?>
                    </div>
                </div>
            </div>
            <?php
        endif;
    ?>
    <?php
    }

    /** Query Args */
    protected function query_args() {
        $settings = $this->get_settings();

        $post_type = $args['post_type'] = $settings['posts_post_type'];
        $args['orderby'] = $settings['posts_orderby'];
        $args['order'] = $settings['posts_order'];
        $args['ignore_sticky_posts'] = 1;
        $args['post_status'] = 'publish';
        $args['offset'] = $settings['posts_offset'];
        $args['posts_per_page'] = $settings['ticker_post_count'];
        $args['post__not_in'] = $post_type == 'post' ? $settings['posts_exclude_posts'] : [];
        $args['tax_query'] = [];

        $taxonomies = get_object_taxonomies($post_type, 'objects');

        foreach ($taxonomies as $object) {
            $setting_key = 'posts_' . $object->name . '_ids';

            if (!empty($settings[$setting_key])) {
                $args['tax_query'][] = [
                    'taxonomy' => $object->name,
                    'field' => 'term_id',
                    'terms' => $settings[$setting_key],
                ];
            }
        }

        return $args;
    }

}
